/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor.intrusiondetection.reference
{
    using org.owasp.appsensor;
    using org.owasp.appsensor.errors;
    using org.owasp.appsensor.intrusiondetection;
    using System.Collections.Generic;
    using System;


/**
 * This is the default representation of the Intrusion Storage 
 * mechanism.  This implementation is very simple in that it 
 * stores all of the intrusions in memory in a HashMap.  
 * <p>
 * The intrusion store stores IntrusionRecord's on a per-user 
 * basis.  Each user has an IntrusionRecord, and if one does not
 * yet exist, it will be automatically created when an exception 
 * is added for this user.  The unique identifier used to determine
 * the user is the account id. 
 * Note: This implementation does include a default system account
 * that is used when exceptions are added that are not user-specific, 
 * ie. the exception is of type AppSensorSystemException.  A similar
 * mechanism is recommended for alternative implementations.
 * <p>
 * Certainly, more complex implementations are possible.
 * Some likely examples include: 
 * <ul>
 * 	<li>File Storage</li>
 * 	<li>DB Storage</li>
 * 	<li>Cloud Storage</li>
 * 	<li>etc.</li>
 * </ul> 
 * 
 * @author Michael Coates (michael.coates .at. owasp.org) 
 *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton (jtmelton .at. gmail.com)
 *         <a href="http://www.jtmelton.com/">jtmelton</a>
 * @since February 24, 2010
 * @see org.owasp.appsensor.intrusiondetection.IntrusionStore
 */
public class DefaultIntrusionStore : IntrusionStore {
	
	private static volatile IntrusionStore singletonInstance;

    public static IntrusionStore getInstance()
    {
        if ( singletonInstance == null ) {
            lock ( typeof(DefaultIntrusionStore) ) {
                if ( singletonInstance == null ) {
                    singletonInstance = new DefaultIntrusionStore();
                }
            }
        }
        return singletonInstance;
    }
	
	/** The hashmap for storing intrusions */
    //jtm - 5/23/2011 - change to concurrent map to address http://code.google.com/p/appsensor/issues/detail?id=6
    private static Dictionary<String, IntrusionRecord> intrusionStore = new Dictionary<String, IntrusionRecord>();
	
	/**
	 * {@inheritDoc}
	 */
	public AppSensorIntrusion addExceptionToIntrusionStore(Exception e) {		
		//check the exception when adding it to the intrusion store.
		//if it is a system exception (AppSensorSystemException), use the
		//system user id to store it, otherwise use the current user
		
		IntrusionRecord intrusionRecord;
		if (e is AppSensorSystemException) {
			//get intrusion record for system user
			intrusionRecord = getIntrusionRecordForSystemUser();
		} else {
			//get intrusion record for current user 
			intrusionRecord = getIntrusionRecordForCurrentUser();
		}
		
		//create AppSensorIntrusion container that will initialize all necessary fields
		AppSensorIntrusion asi = new AppSensorIntrusion(e);
		
		// add intrusion to air record 
		intrusionRecord.AddIntrusionToRecord(asi);
		return asi;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IntrusionRecord getIntrusionRecordForCurrentUser() {
		//get current user
		ASUser user = APPSENSOR.AsUtilities.CurrentUser;
		return getIntrusionRecordForUser(user);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IntrusionRecord getIntrusionRecordForSystemUser() {
		ASUser user = new AppSensorSystemUser();
		return getIntrusionRecordForUser(user);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IntrusionRecord getIntrusionRecordForUser(ASUser user) {
		String userId = user.AccountId.ToString();
		IntrusionRecord intrusionRecord = null;
		// Check for existing record
		if (! intrusionStore.ContainsKey(userId)) {
			//create new record if none found
			intrusionRecord = new DefaultIntrusionRecord(userId);
			
			//add newly created record to store
			if ( !intrusionStore.ContainsKey(userId) ) intrusionStore.Add(userId, intrusionRecord);
		} else {
			//found a record - just return it
			intrusionRecord = intrusionStore[userId];
		}

		return intrusionRecord;
	}

	/**
	 * {@inheritDoc}
	 */
	public List<IntrusionRecord> getAllIntrusionRecords() {
		List<IntrusionRecord> allRecords = new List<IntrusionRecord>();
		foreach (String key in  intrusionStore.Keys) {
			allRecords.Add(intrusionStore[key]);
		}
		return allRecords;
	}

}
}